//
// $Id: VariableProbe.cpp,v 1.8 2007/01/09 17:43:02 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2002-2007, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "VariableProbe.h"

//****************************************************************************************//
//								VariableProbe Class										  //	
//****************************************************************************************//
VariableProbe *VariableProbe::instance = NULL;

VariableProbe::VariableProbe() {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	//
	// -----------------------------------------------------------------------
}

VariableProbe::~VariableProbe() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Do nothing for now
	//
	// -----------------------------------------------------------------------
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Public Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
AbsProbe* VariableProbe::Instance() {

	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//	Ensure that the VariableProbe is a singleton.
	// -----------------------------------------------------------------------

	// Use lazy initialization
	if(instance == NULL) 
		instance = new VariableProbe();

	return instance;	
}

ItemVector* VariableProbe::CollectItems(Object *object) {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Run the variable probe
	// -----------------------------------------------------------------------
	ItemVector *collectedItems = new ItemVector();

	// get the varRef from the provided object
	ObjectEntity* varRef = object->GetElementByName("var_ref");

	// check datatypes - only allow string
	if(varRef->GetDatatype() != OvalEnum::STRING_TYPE) {
		throw ProbeException("Error: invalid data type specified on var_ref. Found: " + OvalEnum::DatatypeToString(varRef->GetDatatype()));
	}

	// check operation - only allow  equals, not equals and pattern match
	if(varRef->GetOperation() != OvalEnum::EQUALS_OPERATION 
		&& varRef->GetOperation() != OvalEnum::PATTERN_MATCH_OPERATION 
		&& varRef->GetOperation() != OvalEnum::NOT_EQUAL_OPERATION) {
		throw ProbeException("Error: invalid operation specified on var_ref. Found: " + OvalEnum::OperationToString(varRef->GetOperation()));
	}

	if(varRef->GetVarRef() == NULL) {
		if(varRef->GetOperation() == OvalEnum::EQUALS_OPERATION) {
			
			// get an item for this var ref
			Item* item = this->GetItemForVarId(varRef->GetValue());
			if(item != NULL) {
				collectedItems->push_back(item);
			}

		} else {

			//
			// handle operation of not equal or pattern match
			// 

			StringVector* varIds = this->GetVariableIds();

			// loop through all vars
			StringVector::iterator iterator;
			for(iterator = varIds->begin(); iterator != varIds->end(); iterator++) {

				if(varRef->GetOperation() == OvalEnum::NOT_EQUAL_OPERATION) {
					if(varRef->GetValue().compare((*iterator)) != 0) {
						
						Item* item = this->GetItemForVarId((*iterator));
						if(item != NULL) {
							collectedItems->push_back(item);
						}
					}
				} else {
					if(this->myMatcher->IsMatch(varRef->GetValue().c_str(), (*iterator).c_str())) {
						Item* item = this->GetItemForVarId((*iterator));
						if(item != NULL) {
							collectedItems->push_back(item);
						}
					}
				}
				
			}
			delete varIds;
		}
	} else {

		StringVector* varIds = this->GetVariableIds();
	
		// loop through all varIds in the definitions document
		// only keep varIds that match operation and value and var check
		StringVector::iterator it;
		for(it = varIds->begin(); it != varIds->end(); it++) {
			ItemEntity* tmp = new ItemEntity("var_ref", (*it), OvalEnum::STRING_TYPE, true, OvalEnum::EXISTS_STATUS);
			if(varRef->Analyze(tmp) == OvalEnum::TRUE_RESULT) {
				Item* item = this->GetItemForVarId((*it));
				if(item != NULL) {
					collectedItems->push_back(item);
				}
			}
			delete tmp;
			tmp = NULL;
		}
		delete varIds;
	}

	return collectedItems;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Private Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
Item* VariableProbe::CreateItem() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Return a new Item created for storing environment variable information
	//
	// -----------------------------------------------------------------------

	Item* item = new Item(0, 
						"http://oval.mitre.org/XMLSchema/oval-system-characteristics-5#independent", 
						"ind-sc", 
						"http://oval.mitre.org/XMLSchema/oval-system-characteristics-5#independent independent-system-characteristics-schema.xsd", 
						OvalEnum::ERROR_STATUS, 
						"variable_item");

	return item;
}

StringVector* VariableProbe::GetVariableIds() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Return all the variable ids in the definitions file being processed.
	// -----------------------------------------------------------------------

	DOMElement* variablesElm = XmlCommon::FindElementNS(DocumentManager::GetDefinitionDocument(), "variables");

	StringVector* varIds = new StringVector();
		
	if(variablesElm != NULL) {

		// loop through all child elements
		DOMNodeList *vars = variablesElm->getChildNodes();
		unsigned int index = 0;
		while(index < vars->getLength()) {
			DOMNode *tmpNode = vars->item(index);

			//	only concerned with ELEMENT_NODEs
			if (tmpNode->getNodeType() == DOMNode::ELEMENT_NODE) {
				DOMElement *var = (DOMElement*)tmpNode;

				//	get the id
				string id = XmlCommon::GetAttributeByName(var, "id");
				varIds->push_back(id);
			}

			index ++;
		}

	} else {
		throw ProbeException("Error: Variable probe unable to locate any varaibles in the current oval definitions docuemnt.");
	}

	return varIds;
}


Item* VariableProbe::GetItemForVarId(string varId) {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Return an Item for the specified variable id
	// -----------------------------------------------------------------------

	Item* item = NULL;
		
	AbsVariable* var = VariableFactory::GetVariable(varId);

	VariableValueVector* varValues = var->GetValues();

	// loop through all values
	VariableValueVector::iterator iterator;
	for(iterator = varValues->begin(); iterator != varValues->end(); iterator++) { 
		if(item == NULL) {
			item = this->CreateItem();
			item->SetStatus(OvalEnum::EXISTS_STATUS);
			item->AppendElement(new ItemEntity("var_ref", varId, OvalEnum::STRING_TYPE, true, OvalEnum::EXISTS_STATUS));
		}
		string value = (*iterator)->GetValue();	
		item->AppendElement(new ItemEntity("value", value, OvalEnum::STRING_TYPE, false, OvalEnum::EXISTS_STATUS));
	}

	return item;
}
