//
// $Id: FileHashProbe.cpp,v 1.3 2007/01/16 20:09:26 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2002-2007, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "FileHashProbe.h"

//****************************************************************************************//
//								FileHashProbe Class										  //	
//****************************************************************************************//
FileHashProbe* FileHashProbe::instance = NULL;

FileHashProbe::FileHashProbe() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//	Note that this is a private constructor
	// -----------------------------------------------------------------------

}

FileHashProbe::~FileHashProbe() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//	Do nothing for now
	// -----------------------------------------------------------------------

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Public Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
AbsProbe* FileHashProbe::Instance() {

	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//	Ensure that the FileHashProbe is a singleton.
	// -----------------------------------------------------------------------

	// Use lazy initialization
	if(instance == NULL) 
		instance = new FileHashProbe();

	return instance;	
}

ItemVector* FileHashProbe::CollectItems(Object* object) {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//	Get all the files on the system that match the pattern. 
	//
	// -----------------------------------------------------------------------

	ItemVector *collectedItems = new ItemVector();

	// get the path and file name
	ObjectEntity* path = object->GetElementByName("path");
	ObjectEntity* fileName = object->GetElementByName("filename");

	FileFinder fileFinder;
	StringPairVector* filePaths = fileFinder.SearchFiles(path, fileName, object->GetBehaviors());

	// Loop through all file paths
	while(filePaths->size() > 0) {
		StringPair* fp = filePaths->at(filePaths->size()-1);
		filePaths->pop_back();

		// build the path string
		string filePath = fp->first;
		if(filePath[filePath.length()-1] != Common::fileSeperator)
			filePath.append(1, Common::fileSeperator);

		if(fp->second[0] != Common::fileSeperator) {
			filePath.append(fp->second);
		} else {
			filePath.append(fp->second.substr(1, fp->second.length()-2));
		}

		// create a new item
		Item *item = NULL;
		item = this->CreateItem();
		item->SetStatus(OvalEnum::EXISTS_STATUS);
		item->AppendElement(new ItemEntity("path", fp->first, OvalEnum::STRING_TYPE, true, OvalEnum::EXISTS_STATUS));
		item->AppendElement(new ItemEntity("filename", fp->second, OvalEnum::STRING_TYPE, true, OvalEnum::EXISTS_STATUS));

		// call the hashing functions
		this->GetMd5(filePath, item);
		this->GetSha1(filePath, item);

		if(item != NULL) {
			collectedItems->push_back(item);
		}
		item = NULL;

		delete fp;
	}

	return collectedItems;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Private Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
Item* FileHashProbe::CreateItem() {
	// -----------------------------------------------------------------------
	//
	//  ABSTRACT
	//
	//  Return a new Item created for storing file hash information
	//
	// -----------------------------------------------------------------------

	Item* item = new Item(0, 
						"http://oval.mitre.org/XMLSchema/oval-system-characteristics-5#independent", 
						"ind-sc", 
						"http://oval.mitre.org/XMLSchema/oval-system-characteristics-5#independent independent-system-characteristics-schema.xsd", 
						OvalEnum::ERROR_STATUS, 
						"filehash_item");

	return item;
}

void FileHashProbe::GetMd5(string filePath, Item* item) {

	try {

		//////////////////////////////////////////////////////
		////////////////////////  MD5  ///////////////////////
		//////////////////////////////////////////////////////

		char buf[1024];
		FILE* fp = NULL;
		fp = fopen(filePath.c_str(), "r");
		if (fp == NULL) {
			string errorMessage = "(FileHashProbe) Unable to get MD5 information for the file '";
			errorMessage.append(filePath);
			errorMessage.append("'");
			item->AppendElement(new ItemEntity("md5", buf, OvalEnum::STRING_TYPE, false, OvalEnum::ERROR_STATUS));
			item->AppendMessage(new OvalMessage(errorMessage));
		
		} else {
			// Create the md5 hash.  This constructor creates a new md5 object, updates the hash,
			// finalizes the hash, and closes the FILE object.
			
			MD5 context(fp);

			memset(buf, '\0', sizeof(buf));
			SNPRINTF(buf, sizeof(buf)-1, "%s", context.hex_digest());
			buf[sizeof(buf)-1] = '\0';
			item->AppendElement(new ItemEntity("md5", buf, OvalEnum::STRING_TYPE, false, OvalEnum::EXISTS_STATUS));
		}

		//////////////////////////////////////////////////////
		//////////////////////////////////////////////////////
	} catch(ProbeException ex) {	
	
		Log::Debug(ex.GetErrorMessage());

	} catch(...) {	
	
		string errMsg = "";
		errMsg.append("(FileMd5Probe) Unknown error attempting to get md5 information for the file '");
		errMsg.append(filePath);
		errMsg.append("'");
		Log::Debug(errMsg);		
	}
}

void FileHashProbe::GetSha1(string filePath, Item* item) {
	
	// TODO: need to find an implementation of the sha1 algorythm and implement this function.
	
	item->AppendElement(new ItemEntity("sha1", "", OvalEnum::STRING_TYPE, false, OvalEnum::NOT_COLLECTED_STATUS));
	item->AppendMessage(new OvalMessage("sha1 hashing of files is not currently supported."));
}
