//
// $Id: Item.h,v 1.21 2006/05/01 13:27:33 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2006, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#ifndef ITEM_H
#define ITEM_H

#include <string>
#include <vector>
#include <map>
#include <utility>
#include <iostream>
#include <stdlib.h>

//	required xerces includes
#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMAttr.hpp>
#include <xercesc/dom/DOMDocument.hpp>
#include <xercesc/dom/DOMNodeList.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/parsers/AbstractDOMParser.hpp>

#include "ItemEntity.h"
#include "OvalMessage.h"
#include "Exception.h"
#include "DocumentManager.h"


XERCES_CPP_NAMESPACE_USE
using namespace std;

class Item;

/**
	A vector for storing Item objects. 
	Stores only pointers to the objects. 
*/
typedef vector < Item*, allocator<Item*> > ItemVector;

/**	
	A pair for storing  item ids and Item objects together. 
	Stores only pointers to the objects. 
*/
typedef pair <int, Item* > ItemPair;

/**	
	A map for storing ItemPairs. 
	Stores only pointers to the objects. 
*/
typedef map <int, Item* > ItemMap;

/**
	This class represents an Item in a sytem characteristics file.
    Items must be unique. The Item::Equals() method and the caching methods
	provided in this class should be utilized to ensure uniqueness. This class also
	provides staticv methods for assigning Item ids.
*/
class Item {
public:
	Item(int id = 0, string xmlns = "", string xmlnsAlias = "", string schemaLocation = "", OvalEnum::SCStatus status = OvalEnum::EXISTS_STATUS, string name = "");
	~Item();

	static int AssignId();

	bool Equals(Item* Item);
	ItemEntityVector* GetElementsByName(string elementName);
	void Parse(DOMElement* scItemElm);
	void Write(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* scFile, DOMElement* itemsElm);

	ItemEntityVector* GetElements();
	void SetElements(ItemEntityVector* elements);

	int GetId();
	void SetId(int id);

	void SetIsWritten(bool isWritten);
	bool GetIsWritten();

	OvalMessageVector* GetMessages();
	void SetMessages(OvalMessageVector* messages);

	string GetName();
	void SetName(string name);

	string GetSchemaLocation();
	void SetSchemaLocation(string schemaLocation);

	string GetXmlns();
	void SetXmlns(string xmlns);

	string GetXmlnsAlias();
	void SetXmlnsAlias(string xmlnsAlias);

	OvalEnum::SCStatus GetStatus();
	void SetStatus(OvalEnum::SCStatus status);

	void AppendElement(ItemEntity* itemEntity);
	void AppendMessage(OvalMessage* msg);

	ItemEntityVector* GetObjectElements();
    
	static Item* SearchCache(int id);
	static void ClearCache();
	static void Cache(Item* item);

private:
	void DeleteElements();

	ItemEntityVector elements;
	int id;
	OvalMessageVector messages;
	string name;
	OvalEnum::SCStatus status;
	string xmlns;
	string xmlnsAlias;
	string schemaLocation;
	bool isWritten;

	static int idCounter;

	static ItemMap processedItemsMap;
};

/** 
	This class represents an Exception that occured while processing an item.
*/
class ItemException : public Exception {
	public:
		ItemException(string errMsgIn = "", int severity = ERROR_FATAL, Exception *ex = NULL);
		~ItemException();
};

#endif
