//
// $Id: Criterion.cpp,v 1.10 2006/06/21 16:37:53 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2006, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "Criterion.h"

//****************************************************************************************//
//								Criterion Class											  //	
//****************************************************************************************//
Criterion::Criterion(bool negate, OvalEnum::ResultEnumeration result, Test* testRef) 
					: AbsCriteria(negate, result) {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Create a compelete Criterion object
	//
	// -----------------------------------------------------------------------

	this->SetTestRef(testRef);
}

Criterion::~Criterion() {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	//
	// -----------------------------------------------------------------------
}

// ***************************************************************************************	//
//								 Public members												//
// ***************************************************************************************	//

Test* Criterion::GetTestRef() {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Return the testRef field's value
	//
	// -----------------------------------------------------------------------

	return this->testRef;
}

void Criterion::SetTestRef(Test* testRef) {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Return the testRef field's value
	//
	// -----------------------------------------------------------------------

	this->testRef = testRef;
}

void Criterion::Write(DOMElement* parentElm) {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	writes a criterion element
	//	calls test->Write() on the test ref
	//
	// -----------------------------------------------------------------------

	// get the parent document
	XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* resultDoc = parentElm->getOwnerDocument();

	// create a new criteria element
	DOMElement* criterionElm = XmlCommon::AddChildElement(resultDoc, parentElm, "criterion");

	// add the attributes
	XmlCommon::AddAttribute(criterionElm, "test_ref", this->GetTestRef()->GetId());

	XmlCommon::AddAttribute(criterionElm, "version", Common::ToString(this->GetTestRef()->GetVersion()));

	if(this->GetTestRef()->GetVariableInstance() != 1) {
		XmlCommon::AddAttribute(criterionElm, "variable_instance", Common::ToString(this->GetTestRef()->GetVariableInstance()));
	}

	string negateStr = Common::ToString(this->GetNegate());
	if(negateStr.compare("true") == 0) {
		XmlCommon::AddAttribute(criterionElm, "negate", negateStr);
	}

	XmlCommon::AddAttribute(criterionElm, "result", OvalEnum::ResultToString(this->GetResult()));

	// write the test ref
	this->GetTestRef()->Write(Analyzer::GetResultsTestsElm());
}

void Criterion::Parse(DOMElement* criterionElm) {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	parses Criterion elm to a Criterion obj
	//	calls test->Parse() on the Test Ref
	//
	// -----------------------------------------------------------------------

	// get the attributes
	string negateStr = XmlCommon::GetAttributeByName(criterionElm, "negate");
	if(negateStr.compare("") == 0 || negateStr.compare("false") == 0) {
        this->SetNegate(false);
	} else {
	    this->SetNegate(true);
	}

	string testRefStr = XmlCommon::GetAttributeByName(criterionElm, "test_ref");
	Test* testRef = NULL;
	testRef = Test::SearchCache(testRefStr);
	if(testRef == NULL) {
		DOMElement* testsElm = XmlCommon::FindElementNS(criterionElm->getOwnerDocument(), "tests");
		DOMElement* testElm = XmlCommon::FindElementByAttribute(testsElm, "id", testRefStr);	
		testRef = new Test();
		testRef->Parse(testElm);
	}
	this->SetTestRef(testRef);
}

OvalEnum::ResultEnumeration Criterion::Analyze() {
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	calls test->Analyze()
	//	applies negate attribute
	//	saves and return the result
	// -----------------------------------------------------------------------

	// analyze the test
	OvalEnum::ResultEnumeration currentResult = this->GetTestRef()->Analyze();

	// apply the negate attribute and save the result
	if(this->GetNegate()) {
		this->SetResult(OvalEnum::NegateResult(currentResult));
	} else {
		this->SetResult(currentResult);
	}

	return this->GetResult();
}
