//
// $Id: CompoundTestAnalyzer.cpp,v 1.12 2005/08/05 20:19:15 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2005, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "CompoundTestAnalyzer.h"

//****************************************************************************************//
//							CompoundTestAnalyzer Class									  //	
//****************************************************************************************//
CompoundTestAnalyzer *CompoundTestAnalyzer::instance = NULL;

CompoundTestAnalyzer::CompoundTestAnalyzer() : Analyzer()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	
	// -----------------------------------------------------------------------

}

CompoundTestAnalyzer::~CompoundTestAnalyzer()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//
	// -----------------------------------------------------------------------
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Public Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
Analyzer* CompoundTestAnalyzer::Instance()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Force this class to be a singleton. Return an Instance of the 
	//	DefiitionAnalyzer
	//
	// -----------------------------------------------------------------------

	if(instance == NULL)
		instance = new CompoundTestAnalyzer();

	return instance;
}

Result CompoundTestAnalyzer::Run(DOMElement *test)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Recursively evaluate compound tests. Compound tests may consist of other
	//	compound tests. Before a given compound test can be evaluated the sub tests
	//	that make up the compound test must be evaluated. This will be done 
	//	recursively as follows:
	//		0- If already processed return
	//		1- Loop through the subtests
	//			 - Run an analyzer for each test to get the result
	//		2- Store the result of each subtest
	//		3- Combine the subtests results based on the operation to get the result
	//		4- Store the test id and result
	//		5- Write the result to the result document
	//		6- Return the result
	// -----------------------------------------------------------------------

	Result compoundTestResult = unknown_result;

	// Get a reference to the AnalyzerFactory
	AnalyzerFactory *analyzerFactory = AnalyzerFactory::Instance();

	// Get the test id
	string testId = XmlCommon::GetAttributeByName(test, "id");

	// Check if the test has already been processed.
	if(!Processed(testId)) {

		iVector subtestResults;

		// Get the comment
		string comment = XmlCommon::GetAttributeByName(test, "comment");

		// Get the operation
		string operation = XmlCommon::GetAttributeByName(test, "operation");

		// Add the test to the result doc	
		DOMElement *compoundTest = this->resultWriter->AddCompoundTest(testId, comment, operation);

		// Loop through all the subtests
		for (DOMNode* tmpNode = test->getFirstChild(); tmpNode != NULL; tmpNode=tmpNode->getNextSibling()) {
			if(tmpNode->getNodeType() == DOMNode::ELEMENT_NODE) {
				DOMElement* subtest = (DOMElement*)tmpNode;

				if((XmlCommon::GetElementName((DOMElement*)subtest)).compare("subtest") == 0) {
				
					// Get the negate attribute
					string negateAttr = XmlCommon::GetAttributeByName(subtest, "negate");

					// Get the test_ref attribute
					string test_refAttr = XmlCommon::GetAttributeByName(subtest, "test_ref");

					//	Get the corresponding test
					DOMNode *testNode = XmlCommon::FindNodeByAttribute(XmlCommon::FindNodeNS((ovalDoc)->getDocumentElement(), "tests"), "id", test_refAttr);
					if(testNode == NULL)
						throw AnalyzerException("Unable to locate the test that corresponds to the current criterion. id: " + test_refAttr);

					//	Get an Analyzer for this test
					Analyzer *testAnalyzer = analyzerFactory->GetAnalyzer((DOMElement*)testNode);

					//	Run the analyzer
					Result subtestResult = testAnalyzer->Run((DOMElement*)testNode);

					//	Write the subtest to the results file
					this->resultWriter->AddSubtest(compoundTest, negateAttr, test_refAttr, ResultToString(subtestResult));

					// Use the negate attribute
					if(negateAttr.compare("true") == 0) 
						subtestResult = NegateResult(subtestResult);

					// Add the result to the vector of results			 
					subtestResults.push_back(subtestResult);
				}
			}

		}

		//	Compute the combined result
		compoundTestResult = CombineResultsByOperation(&subtestResults, operation);

		// Set the result for the compound test in the result document
		resultWriter->SetResult(compoundTest, ResultToString(compoundTestResult));

		// Store the test id and result
		Store(testId, compoundTestResult);

	} else {
		compoundTestResult = this->GetTestResult(testId);
	}

	return compoundTestResult;
}
