//
// $Id: ProcessData.cpp,v 1.13 2005/08/05 20:47:16 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2005, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "ProcessData.h"

//****************************************************************************************//
//								ProcessData Class										  //	
//****************************************************************************************//
// Initialize the item Vector
sVector ProcessData::items;

ProcessData::ProcessData(DOMElement *test)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Initialize a new ProcessData and populate with the node's data
	// -----------------------------------------------------------------------

	string testName = XmlCommon::GetElementName(test);
	if(testName.compare("process_test") != 0)
		throw ProbeDataException("Error: ProcessData() Invalid test node specified. Received: " + testName);


	//	Init the data members
	command = new ObjectElement();
	exec_time = new DataElement();
	pid = new DataElement();
	ppid = new DataElement();
	priority = new DataElement();
	scheduling_class = new DataElement();
	start_time = new DataElement();
	tty = new DataElement();
	user_id = new DataElement();

	//	Call the parse node function
	ParseNode(test);

	// set the xmlns and schema location
	this->SetXmlnsAlias("unix");
	this->SetXmlns("http://oval.mitre.org/XMLSchema/system_characteristics#unix");
	this->SetSchemaLocation("http://oval.mitre.org/XMLSchema/system_characteristics#unix unix-system-characteristics-schema.xsd");
}

ProcessData::ProcessData()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Initialize a new ProcessData as an empty object.
	// -----------------------------------------------------------------------

    command = new ObjectElement();
	exec_time = new DataElement();
	pid = new DataElement();
	ppid = new DataElement();
	priority = new DataElement();
	scheduling_class = new DataElement();
	start_time = new DataElement();
	tty = new DataElement();
	user_id = new DataElement();

	// set the xmlns and schema location
	this->SetXmlnsAlias("unix");
	this->SetXmlns("http://oval.mitre.org/XMLSchema/system_characteristics#unix");
	this->SetSchemaLocation("http://oval.mitre.org/XMLSchema/system_characteristics#unix unix-system-characteristics-schema.xsd");
}

ProcessData::~ProcessData()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Delete data members
	// -----------------------------------------------------------------------

	delete command;
	delete exec_time;
	delete pid;
	delete ppid;
	delete priority;
	delete scheduling_class;
	delete start_time;
	delete tty;
	delete user_id;
}

// ***************************************************************************************	//
//								Public members												//
// ***************************************************************************************	//
void ProcessData::Write()
{
  // -----------------------------------------------------------------------
  //	Abstract
  //
  //	Create a new inetlisteningservers_test node and add it 
  //	to the dataDocument.
  //	
  // -----------------------------------------------------------------------

  	// Create a string representation of the item
	string item = command->object;

	// Ensure that the item has not already been written
	if(WroteItem(&items, item))
		return;

	//	Create a new item node
	DOMElement *processItem = CreateNewItemNode(dataDocument, "process_item");

	// Add the object element
	DOMElement *objectElm = XmlCommon::AddChildElement(dataDocument, processItem, "object");

	// Add the status attribute to the object if not default value
	string strStatus = this->SwitchStatusType(this->objectStatus);
	if (strStatus.compare("exists") != 0)
		XmlCommon::AddAttribute(objectElm, "status", strStatus);

	//	Add the program_name element
	AddObjectChildElement(dataDocument, objectElm, "command", command);		

	// Add the data section if the object status is exists
	if(objectStatus == exists && !this->isPatternMatchObject) {
		DOMElement *dataElm = XmlCommon::AddChildElement(dataDocument, processItem, "data");

		// Add data subelements
		AddDataChildElement(dataDocument, dataElm, "exec_time", this->exec_time);
		AddDataChildElement(dataDocument, dataElm, "pid", this->pid);
		AddDataChildElement(dataDocument, dataElm, "ppid", this->ppid);
		AddDataChildElement(dataDocument, dataElm, "priority", this->priority);
		AddDataChildElement(dataDocument, dataElm, "scheduling_class", this->scheduling_class);
		AddDataChildElement(dataDocument, dataElm, "start_time", this->start_time);
		AddDataChildElement(dataDocument, dataElm, "tty", this->tty);
		AddDataChildElement(dataDocument, dataElm, "user_id", this->user_id);
	}

	// Add the new item to the data document
	InsertItem(dataDocument, processItem, "process_items");

}
// ***************************************************************************************	//
//								Private members												//
// ***************************************************************************************	//
void ProcessData::ParseNode(DOMElement *test)
{
  // -----------------------------------------------------------------------
  //	Abstract
  //
  //	Parse the node gathering all data to populate this process object. 
  // -----------------------------------------------------------------------

	///////////////////////////////////////////////////////////////////////////
	//	Get the test id
	///////////////////////////////////////////////////////////////////////////
	testId = XmlCommon::GetAttributeByName(test, "id");
	if(testId.compare("") == 0)
		throw ProbeDataException("Error: Unable to find the 'id' attribute for a Red Hat Process test.");	

  	///////////////////////////////////////////////////////////////////////////
	//	Parse the node
	///////////////////////////////////////////////////////////////////////////
	// Get the object element
	DOMElement *object = XmlCommon::FindNodeNS(test, "object");
	if (object == NULL) {
		throw new ProbeDataException("Error: Unable to locate object section for test: " + testId + "\n");
	}

	// Loop over childern of the object node
	//	get a list of the child nodes and their values
	string childName	= "";
	string childValue	= "";
	string childOp		= "";	

	DOMNodeList *objectChildren = object->getChildNodes();
	unsigned int index = 0;
	while(index < objectChildren->getLength()) {
		DOMNode *tmpNode = objectChildren->item(index);

		//	only concerned with ELEMENT_NODEs
		if (tmpNode->getNodeType() == DOMNode::ELEMENT_NODE) {
			DOMElement* objectChild = (DOMElement*)tmpNode;
			//	get the name of the child
			childName = XmlCommon::GetElementName(objectChild);
			childValue = XmlCommon::GetDataNodeValue(objectChild);
			childOp = XmlCommon::GetAttributeByName(objectChild, "operator");
			
			// Convert the type
			objectTypes curType;
			if(childOp.compare("pattern match") == 0) {
				curType = pattern_match;
				this->isPatternMatchObject = true;
			} else {
				curType = literal;
			}
			
			//	Get the command value	
			if(childName.compare("command") == 0) {

				command->object = childValue;
				command->type = curType;
			}
		}
		index ++;
	}
}
