<?xml version="1.0" encoding="UTF-8"?>
<!--
	AUTHOR:Matt Burton, The Mitre Corporation 
	DATE: 02May05 
	
	The results_to_html stylesheet takes the results of a system's OVAL scan and displays it in a table with pertinent information
	to the scan such as all OVAL ID, CVE ID, Descriptions, OVAL version & status of the vulnerabilities checked.  
	The table is color coded, with a key at the top of the page, to group the vulnerabilities into three separate 
	categories of (1) Vulnerable, (2) Not Vulnerable, and (3) Unknown.
-->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
	xmlns:oval_results="http://oval.mitre.org/XMLSchema/oval_results">
	<xsl:output method="html"/>
	
	<!-- Style declarations for pretty formatting -->
	<xsl:template name="oval_results_style">
		<style type="text/css">
			<xsl:comment>
TD.title {BACKGROUND-COLOR: #000000; COLOR: #ffc; FONT-WEIGHT: bold; TEXT-ALIGN: left} TD.label {BACKGROUND-COLOR: #99cc99}
TR.vulnerable {BACKGROUND-COLOR: #FF4040} TR.notvulnerable {BACKGROUND-COLOR: #8DB6CD} TR.unknown {BACKGROUND-COLOR: #FFFF99}
TD.vulnerable {BACKGROUND-COLOR: #FF4040; text-align: center} TD.notvulnerable {BACKGROUND-COLOR: #8DB6CD; text-align: center} TD.unknown {BACKGROUND-COLOR: #FFFF99; text-align: center}
TD.label2 {FONT-WEIGHT: bold}
			</xsl:comment>
		</style>
	</xsl:template>
	
	<!-- Gets the top level node -->
	<xsl:template match="oval_results:oval_results">
		<HTML>
			<HEAD>
				<TITLE> OVAL Results </TITLE>
				<!-- Get stylesheet -->
				<xsl:call-template name="oval_results_style"/>
			</HEAD>
			<BODY>
				<!-- Get pertinent information in this order: system_info, generators, definitions --> 
				<xsl:apply-templates select="oval_results:system_info"/>
				<xsl:apply-templates select="oval_results:generators"/>
				<xsl:apply-templates select="oval_results:definitions"/>
			</BODY>
		</HTML>
	</xsl:template>
	
	
	<!-- Get the system_info and put it into a table. -->
	<xsl:template match="oval_results:system_info">
		<table border="1" cellspacing="0" cellpaddign="2" width="100%" bgcolor="#cccccc">
			<tr><td class="title" colspan="2">System Information</td></tr>
			<tr>
				<td class="label2" width="20%">Host Name</td>
				<td width="80%"><xsl:value-of select="oval_results:primary_host_name"/>&#160;</td> <!-- $#160; is used to keep empty cells in the table clean -->
			</tr>
			<tr>
				<td class="label2" width="20%">Operating System</td>
				<td width="80%"><xsl:value-of select="oval_results:os_name"/>&#160;</td>
			</tr>
			<tr>
				<td class="label2" width="20%">Operating System Version</td>
				<td width="80%"><xsl:value-of select="oval_results:os_version"/>&#160;</td>
			</tr>
			<tr>
				<td class="label2" width="20%">Architecture</td>
				<td  width="80%"><xsl:value-of select="oval_results:architecture"/>&#160;</td></tr>
			<tr><td class="label2" colspan="2">Interfaces</td></tr>
			<xsl:apply-templates select="oval_results:interfaces"/>	
		</table>
	</xsl:template>

	<!-- Get this interface and put it into the table, using templates to loop through all possible interfaces -->
	<xsl:template match="oval_results:interfaces">
		<xsl:for-each select="oval_results:interface">
			<tr>
				<td class="label2" width="20%">Interface Name</td>
				<td  width="80%"><xsl:value-of select="oval_results:interface_name"/>&#160;</td>
			</tr>
			<tr>
				<td class="label2" width="20%">IP Address</td>
				<td width="80%"><xsl:value-of select="oval_results:ip_address"></xsl:value-of>&#160;</td>
			</tr>
			<tr>
				<td class="label2" width="20%">MAC Address</td>
				<td width="80%"><xsl:value-of select="oval_results:mac_address"></xsl:value-of>&#160;</td>
			</tr>
		</xsl:for-each>
	</xsl:template>
	
	<!-- Crate a table for Generators' and call its children. -->
	<xsl:template match="oval_results:generators">
		<table border="1" cellspacing="0" cellpaddign="2" width="100%" bgcolor="#cccccc">
			<tr><td class="title" colspan="5">Schema Information</td></tr>
			<tr>
				<td class="label">Schema Name</td>
				<td class="label">Schema Version</td>
				<td class="label">Product Name</td>
				<td class="label">Product Version</td>
				<td class="label">Timestamp</td>
			</tr>
			<xsl:apply-templates/>
		</table>
	</xsl:template>
	
	<!-- Templates for each of Generators' children that in turn call a generic table. -->
	<xsl:template match="oval_results:oval">
		<tr>
			<td width="20%">Oval Schema</td>
			<xsl:call-template name="generator_info"/>
		</tr>
	</xsl:template>
	<xsl:template match="oval_results:system_characteristics">
		<tr>
			<td width="20%">System Characteristics Schema</td>
			<xsl:call-template name="generator_info"/>
		</tr>
	</xsl:template>
	<xsl:template match="oval_results:results">
		<tr>
			<td width="20%">Results Schema</td>
			<xsl:call-template name="generator_info"/>
		</tr>
	</xsl:template>
	
	<!-- Each child of Generators (oval, system_characteristics, results) is the same, this template gets their children. -->
	<xsl:template name="generator_info">
		<td><xsl:value-of select="oval_results:schema_version"/>&#160;</td>
		<td><xsl:value-of select="oval_results:product_name"/>&#160;</td>
		<td><xsl:value-of select="oval_results:product_version"/>&#160;</td>
		<td>
			<!--Create variable "MessyNumber" to make time stamp a string and then print it out in a readable version -->
			<xsl:variable name="MessyNumber" select="string(oval_results:timestamp)"/>
			<xsl:value-of select="substring($MessyNumber, 5, 2)"/> <xsl:text>/</xsl:text> <!-- month -->
			<xsl:value-of select="substring($MessyNumber, 7, 2)"/> <xsl:text>/</xsl:text> <!-- day -->
			<xsl:value-of select="substring($MessyNumber, 1, 4)"/> <xsl:text> </xsl:text> <!-- year -->
			<xsl:value-of select="substring($MessyNumber, 9, 2)"/> <xsl:text>:</xsl:text> <!-- hour -->
			<xsl:value-of select="substring($MessyNumber, 11, 2)"/> <xsl:text>:</xsl:text> <!-- minute -->
			<xsl:value-of select="substring($MessyNumber, 13, 2)"/>&#160; <!-- second -->
		</td>
	</xsl:template>
	
	
	
	<xsl:template match="oval_results:definitions">
		<table border="1" cellspacing="0" cellpaddign="2" width="100%">
			<tr><td class="title" colspan="5">Oval Test Results</td></tr>
			<!-- ^^^^^^^^^^^^^^^^^ Table Color Key table ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ -->
			<tr>
					<td colspan="5">	
						<table cellspacing="0" border="0" width="100%">
							<td class="vulnerable" width="33%">Vulnerable</td>
							<td class="notvulnerable" width="33%">Not Vulnerable</td>
							<td class="unknown" width="33%">Unknown</td>
						</table>  
					</td>
			</tr>
<!-- ^^^^^^^^^^^^^^^^^ End of Table Color Key table ^^^^^^^^^^^^^^^^^^^^^^^^^ -->
			<tr>
				<td class="label" align="center">OVAL ID</td>
				<td class="label" align="center">CVE ID</td>
				<td class="label" align="center">Description</td>
				<td class="label" align="center">Status</td>
				<td class="label" align="center">Version</td>
			</tr>
			<xsl:apply-templates>
				<!-- sort vulnerabilities into groups of vulnerable, not vulnerable and unknown -->
				<!--This has been disabled until I figure out how to sort both software & configuration-->
				<xsl:sort select="oval_results:criteria/@result" data-type="number" order="descending"/>
			</xsl:apply-templates>
		</table>
	</xsl:template>
	
	<xsl:template match="oval_results:definition">
	 
		<!-- Start of new row -->
		<tr>
			<xsl:choose>
				<xsl:when test="oval_results:criteria/oval_results:configuration">
					<!-- set the background color for the row according to vulnerable/not vulnerable/unknown status-->
					<xsl:choose>
						<xsl:when test="oval_results:criteria/oval_results:software/@result='1' and oval_results:criteria/oval_results:configuration/@result='1'"> <!-- if vulnerable, sets this color for row -->
							<xsl:attribute name="class">vulnerable</xsl:attribute>
						</xsl:when>
						<xsl:when test="./oval_results:criteria/oval_results:software/@result='0' or oval_results:criteria/oval_results:configuration/@result='0'"> <!-- if not vulnerable, sets this color for row -->
							<xsl:attribute name="class">notvulnerable</xsl:attribute>
						</xsl:when>
						<xsl:when test="./oval_results:criteria/oval_results:software/@result='1' and oval_results:criteria/oval_results:configuration/@result='-1'"> <!-- if not vulnerable, sets this color for row -->
							<xsl:attribute name="class">unknown</xsl:attribute>
						</xsl:when>
						<xsl:when test="./oval_results:criteria/oval_results:software/@result='-1' and oval_results:criteria/oval_results:configuration/@result='1'"> <!-- if not vulnerable, sets this color for row -->
							<xsl:attribute name="class">unknown</xsl:attribute>
						</xsl:when>					
						<xsl:otherwise>
							<xsl:attribute name="class">unknown</xsl:attribute> <!-- if error occurs, selects default 'unknown' color for row-->
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<!-- set the background color for the row according to vulnerable/not vulnerable/unknown status-->
						<xsl:choose>
							<xsl:when test="oval_results:criteria/oval_results:software/@result='1'"> <!-- if vulnerable, sets this color for row -->
								<xsl:attribute name="class">vulnerable</xsl:attribute>
							</xsl:when>
							<xsl:when test="./oval_results:criteria/oval_results:software/@result='0'"> <!-- if not vulnerable, sets this color for row -->
								<xsl:attribute name="class">notvulnerable</xsl:attribute>
							</xsl:when>					
							<xsl:otherwise>
								<xsl:attribute name="class">unknown</xsl:attribute> <!-- if error occurs, selects default 'unknown' color for row-->
							</xsl:otherwise>
						</xsl:choose>
				</xsl:otherwise>
			</xsl:choose>
		
			<td width="100" align="center"> 
				<xsl:text disable-output-escaping='yes'>&lt;a </xsl:text>
					<xsl:text>href="http://oval.mitre.org/oval/definitions/data/oval</xsl:text> <!-- OVAL url -->
					<xsl:value-of select='substring-after(@id,"OVAL")'/>
					<xsl:text>.html" target="_blank"></xsl:text>
					<xsl:text disable-output-escaping="yes">&gt;&#10;</xsl:text>
					<xsl:value-of select="@id"/> <!-- writes OVAL ID to first column -->
				<xsl:text disable-output-escaping="yes">&lt;/a&gt;&#10;</xsl:text>
			</td>
			<td width="150" align="center">
				<!-- creates hyperlink to appropriate CVE site -->
				<xsl:text disable-output-escaping='yes'>&lt;a </xsl:text>
						<xsl:text>href="http://www.cve.mitre.org/cgi-bin/cvename.cgi?name=</xsl:text> <!-- CVE url -->
							<xsl:value-of select="oval_results:reference"/> <!-- template call to get CAN/CVE -->
						<xsl:text>"  target="_blank"></xsl:text>
					<xsl:text disable-output-escaping="yes">&gt;&#10;</xsl:text>
						<xsl:value-of select="oval_results:reference"/>		<!-- template call to get CAN/CVE & print -->
				<xsl:text disable-output-escaping="yes">&lt;/a&gt;&#10;</xsl:text>
			</td>
			<td width="700" align="left"> <!-- sets width and alignment for vulnerability description column -->
			
				<xsl:value-of select="oval_results:description"/> <!-- writes vulnerability description to 3rd column -->
			</td>
			
			<td width="100" align="center"> <!-- sets width and alignment for vulnerability status column -->
			
				<xsl:value-of select="oval_results:status"/> <!-- writes vulnerability status to 4th column -->
			</td>
			
			<td width="100" align="center"> <!-- sets width and alignment for vulnerability version column -->
			
				<xsl:value-of select="oval_results:version"/> <!-- writes vulnerability version to 5th column -->
			</td>
		</tr>
			
	</xsl:template>

	
</xsl:stylesheet>
