//
// $Id: Analyzer.h,v 1.5 2005/03/28 15:59:41 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2005, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//
//
//  file:	Analyzer.h
//  author: Jon Baker	
//	
//****************************************************************************************//
#ifndef __ANALYZER_H__
#define __ANALYZER_H__

//	required xerces includes
#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMAttr.hpp>
#include <xercesc/dom/DOMDocument.hpp>
#include <xercesc/dom/DOMNodeList.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/parsers/AbstractDOMParser.hpp>

// other includes
#include "REGEX.h"
#include "Exception.h"
#include "OvalResultXml.h"


using namespace std;

// An enum to define the allowed result values
enum Result{unknown_result, false_result, true_result};

// An enum to define the allowed check attr values
enum CheckAttr{check_all, check_at_least_one, check_none_exist, check_only_one};


//	Define a vector for storing Probe Data objects
typedef vector < int, allocator<int> > iVector;


//****************************************************************************************//
//	Class : Analyzer
//	A simple class to store a test id paired with its result value. 
//****************************************************************************************//

class TestResult
{
public:
	TestResult();
	TestResult(string i, Result r) : testId(i), result(r){};
	~TestResult();

	// -----------------------------------------------------------------------
	//	Public members
	// -----------------------------------------------------------------------
	string GetTestId(){return testId;};
	void SetTestId(string i){testId = i;};
	Result GetResult(){return result;};
	void SetResult(Result r){ result = r;};

private:
	// -----------------------------------------------------------------------
	//	Private data
	// -----------------------------------------------------------------------
	string testId;
	Result result;
};

//	Define a vector for storing TestResult objects
typedef vector < TestResult*, allocator<TestResult*> > trVector;


//****************************************************************************************//
//	Class : Analyzer
//	The Analyzer class will act as an abstract base class from which all
//	all other specifc types of Analyzer will inherit. 
//****************************************************************************************//

class Analyzer  
{
public:
	// -----------------------------------------------------------------------
	//	Destructor
	// -----------------------------------------------------------------------
	virtual ~Analyzer();

	// -----------------------------------------------------------------------
	//	Public members
	// -----------------------------------------------------------------------
	static void Init(OvalResultXml*, XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, bool, string);
	virtual Result Run(DOMElement*) = 0;

protected:
	// -----------------------------------------------------------------------
	//	Constructor
	// -----------------------------------------------------------------------
	Analyzer();

	// -----------------------------------------------------------------------
	//	Protectected members
	// -----------------------------------------------------------------------
	Result CombineResultsByCheck(iVector*, string);
	Result CombineResultsByOperation(iVector*, string);
	Result GetTestResult(string);
	Result NegateResult(Result);
	bool Processed(string);
	string ResultToString(Result result); 
	void Store(string, Result);

	// -----------------------------------------------------------------------
	//	Protectected data
	// -----------------------------------------------------------------------
	REGEX *myMatcher;
	trVector processedTests;

	// Static data for all base Analyzer classes
	static OvalResultXml *resultWriter;
	static XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *dataDoc;
	static XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *mappingDoc;
	static XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *ovalDoc;
	static bool useConfiguration;
	static bool initialized;
	static string family;
};

//	Define an AnalyzerException
class AnalyzerException : public Exception
{
	public:
		AnalyzerException(string errMsgIn = "", int severity = ERROR_FATAL);
		~AnalyzerException();
};

#endif
