//
// $Id: WMIData.cpp,v 1.2 2005/09/06 18:57:55 abuttner Exp $
//
//****************************************************************************************//
// Copyright (c) 2005, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "WMIData.h"

// Initialize the item Vector
sVector WMIData::items;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Class WMIData  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

WMIData::WMIData(DOMElement* testNodeIn)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//	Create an data object and set iits members based on the provided xml node
	//
	//------------------------------------------------------------------------------------//
	
	// Validate the test node.
	string testNodeName = XmlCommon::GetElementName((DOMElement*)testNodeIn);
	if (testNodeName.compare("wmi_test") != 0) {
		throw ProbeDataException("Error: (WMIData) Invalid test node specified.");
	}

	// Initialize the data members.
	wmi_namespace = new ObjectElement();
	wql = new ObjectElement();
	result = new DataElement();

	// Call the ParseNode() function to populate the new WMIData object that was
	// just initialized.
	ParseNode(testNodeIn);

	// set the xmlns and schema location
	this->SetXmlnsAlias("windows");
	this->SetXmlns("http://oval.mitre.org/XMLSchema/system_characteristics#windows");
	this->SetSchemaLocation("http://oval.mitre.org/XMLSchema/system_characteristics#windows windows-system-characteristics-schema.xsd");
}

WMIData::WMIData()
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//	Create an empty data object
	//
	//------------------------------------------------------------------------------------//

	wmi_namespace = new ObjectElement();
	wql = new ObjectElement();
	result = new DataElement();

	// set the xmlns and schema location
	this->SetXmlnsAlias("windows");
	this->SetXmlns("http://oval.mitre.org/XMLSchema/system_characteristics#windows");
	this->SetSchemaLocation("http://oval.mitre.org/XMLSchema/system_characteristics#windows windows-system-characteristics-schema.xsd");
}

WMIData::~WMIData()
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//	Delete data members
	//
	//------------------------------------------------------------------------------------//

	delete wmi_namespace;
	delete wql;
	delete result;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Public Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

void WMIData::Write()
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//	Write a windows WMI data element to the data document.
	//
	//------------------------------------------------------------------------------------//

	// Create a string representation of the item
	string item = wmi_namespace->object + ":" + wql->object;

	// Ensure that the item has not already been written
	if(WroteItem(&items, item))
		return;

	// Create a new item node.
	DOMElement *wmiItem = CreateNewItemNode(dataDocument, "wmi_item");

	// Add the object element
	DOMElement *objectElm = XmlCommon::AddChildElement(dataDocument, wmiItem, "object");

	// Add the status attribute to the object if not default value
	string strStatus = this->SwitchStatusType(this->objectStatus);
	if (strStatus.compare("exists") != 0)
		XmlCommon::AddAttribute(objectElm, "status", strStatus);

	//	Add the namespace element.
	AddObjectChildElement(dataDocument, objectElm, "namespace", this->wmi_namespace);

	//	Add the wql element.
	AddObjectChildElement(dataDocument, objectElm, "wql", this->wql);

	// Add the data section if the object status is exists
	if(objectStatus == exists && !this->isPatternMatchObject) {
		DOMElement *dataElm = XmlCommon::AddChildElement(dataDocument, wmiItem, "data");

		AddDataChildElement(dataDocument, dataElm, "result", this->result);
	}

	// Add the new item to the data document
	InsertItem(dataDocument, wmiItem, "wmi_items");
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Private Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

void WMIData::ParseNode(DOMElement* testNodeIn)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//	Parse the provided DOMNode, gathering all data to populate the current
	//  WMIData object.
	//	
	//------------------------------------------------------------------------------------//

	// Retreive the test id from the DOMNode.
	testId = XmlCommon::GetAttributeByName(testNodeIn, "id");
	if (testId.compare("") == 0) {
		throw ProbeDataException("Error: Unable to find the 'id' attribute for specified the test.");
	}

	///////////////////////////////////////////////////////////////////////////
	//	Parse the node
	///////////////////////////////////////////////////////////////////////////
	// Get the object element
	DOMNode *object = XmlCommon::FindNodeNS(testNodeIn, "object");
	if (object == NULL) {
		throw new ProbeDataException("Error: Unable to locate object section for test: " + testId + "\n");

	}

	// Loop over childern of the object node
	//	get a list of the child nodes and their values
	string childName	= "";
	string childValue	= "";
	string childOp		= "";

	DOMNodeList *objectChildren = object->getChildNodes();
	unsigned int index = 0;
	while(index < objectChildren->getLength())
	{
		DOMNode *tmpNode = objectChildren->item(index);


		//	only concerned with ELEMENT_NODEs
		if (tmpNode->getNodeType() == DOMNode::ELEMENT_NODE) {

			DOMElement *objectChild = (DOMElement*)tmpNode;

			childName = XmlCommon::GetElementName(objectChild);
			childValue = XmlCommon::GetDataNodeValue(objectChild);
			childOp = XmlCommon::GetAttributeByName(objectChild, "operator");

			// Convert the type
			objectTypes curType;
			if(childOp.compare("pattern match") == 0) {
				curType = pattern_match;
				this->isPatternMatchObject = true;
			} else {
				curType = literal;
			}
			
			//	Get the namespace value	
			if(childName.compare("namespace")==0)
			{
				wmi_namespace->object = childValue;
				wmi_namespace->type = curType;
				
			//	Get the wql value
			}else if(childName.compare("wql")==0) {

				wql->object = childValue;
				wql->type = curType;
			}
		}

		index++;
	}
}
	