//
// $Id: ProbeData.h,v 1.13 2005/07/12 17:36:21 bakerj Exp $
//
//****************************************************************************************//
// Copyright (c) 2005, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//
//
//  file:  ProbeData.h
//  author: Jon Baker	
//	
//****************************************************************************************//
#ifndef __PROBEDATA_H__
#define __PROBEDATA_H__

#include "Common.h"
#include "XmlCommon.h"
#include "Exception.h"
#include "REGEX.h"

//	required xerces includes
#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMAttr.hpp>
#include <xercesc/dom/DOMDocument.hpp>
#include <xercesc/dom/DOMNodeList.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/parsers/AbstractDOMParser.hpp>

//	additional includes
#include <string>
#include <vector>

//	namespaces
XERCES_CPP_NAMESPACE_USE
using namespace std;

// An enum to define the allowed types of data
enum objectTypes {literal, pattern_match};

// An enum to define the acceptable data types
enum dataTypes {stringType, integerType, binaryType, booleanType, floatType, versionType};

// An enum to define the acceptable status values
enum statusTypes {error, exists, doesNotExist, notCollected};

//****************************************************************************************//
//	Class : ObjectElement
//	A simple class who's aim is to associate a type(litteral or pattern match) with a string 
//	containing some object specifier.
//****************************************************************************************//

class ObjectElement
{
	public:
	// -----------------------------------------------------------------------
	//	Constructors & Destructors
	// -----------------------------------------------------------------------	
		ObjectElement(objectTypes t, string o) : type(t), object(o) {};
		ObjectElement() {type=literal; object="";};

	// -----------------------------------------------------------------------
	//	Public data
	// -----------------------------------------------------------------------
		objectTypes type;
		string object;
};

//****************************************************************************************//
//	Class : DataElement
//	A simple class who's aim is to associate the data type,and status with a piece of
//  data collected by a probe.
//****************************************************************************************//
class DataElement
{
	public:
	// -----------------------------------------------------------------------
	//	Constructors & Destructors
	// -----------------------------------------------------------------------	
		DataElement(dataTypes t, statusTypes s, string v) : dataType(t), status(s), value(v) {};
		DataElement() {dataType=stringType; status=exists; value="";};

	// -----------------------------------------------------------------------
	//	Public data
	// -----------------------------------------------------------------------
		dataTypes dataType;
		statusTypes status;
		string value;
};

//****************************************************************************************//
//	Class : ProbeData
//	The ProbeData class will act as an abstract base class from which all
//	all other specifc types of probe data will inherit. Each probe has a 
//	ProbeData class that stores the data for the probe, and knows how to write that data
//	to the data document. 
//****************************************************************************************//

class ProbeData  
{
public:
	// -----------------------------------------------------------------------
	//	Destructor
	// -----------------------------------------------------------------------
	virtual ~ProbeData();

	// -----------------------------------------------------------------------
	//	Public members
	// -----------------------------------------------------------------------
	virtual void Write() = 0;
	string GetXmlns();
	string GetXmlnsAlias();
	string GetMessage();
	statusTypes GetStatus();
	string GetTestId();
	string GetSchemaLocation();
	static void ProbeData::SetDataDoc(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *doc);
	void SetMessage(string);
	void SetXmlns(string ns);
	void SetXmlnsAlias(string alias);
	void SetSchemaLocation(string location);
	void SetStatus(statusTypes);

	// Flag to indicate that this is a pattern match object
	bool isPatternMatchObject;

protected:
	// -----------------------------------------------------------------------
	//	Constructor
	// -----------------------------------------------------------------------
	ProbeData();

	// -----------------------------------------------------------------------
	//	Protected members
	// -----------------------------------------------------------------------
	virtual void ParseNode(DOMElement *test) = 0;
	void AddDataChildElement(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, DOMElement*, string, DataElement*);
	void AddObjectChildElement(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, DOMElement*, string, ObjectElement*);
	DOMElement* CreateNewItemNode(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *doc, string testName);
	void InsertItem(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument*, DOMElement*, string);
	string SwitchDataType(dataTypes type);
	string SwitchObjectType(objectTypes type);
	string SwitchStatusType(statusTypes status);
	void UpdateSchemaLocation();
	void UpdateXmlns();
	bool WroteItem(sVector*, string);

	// -----------------------------------------------------------------------
	//	Protected data
	// -----------------------------------------------------------------------

	// manage unique ids for all probe data objects
	static int globalUniqueId;

	// Store a reference to the document to write too
	static XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *dataDocument;

	// the id of his current object
	int objectId;

	// The message associated with a set of probe data.
	string msg;

	// The status flag for the object
	statusTypes objectStatus;

	// The id of the test the dat is being collected for.
	string testId;

	// the namespace to be used when writing this data to xml format
	string xmlns;

	// The string to be used for the schema location when writing this data to xml format
	string schemaLocation;

	// The string to be used as the alias for the specified xmlns
	string xmlnsAlias;
};

//	Define a vector for storing Probe Data objects
typedef vector < ProbeData*, allocator<ProbeData*> > pdVector;

//	Define a ProbeDataException
class ProbeDataException : public Exception
{
	public:

		ProbeDataException(string errMsgIn = "", int severity = ERROR_FATAL);
		~ProbeDataException();
};

#endif
