//
// $Id: FileProbe.cpp,v 1.3 2004/09/02 15:31:02 bakerj Exp $
//
//************************** Property of the MITRE Corporation ***************************//
//
// Copyright (c) 2003 - The MITRE Corporation
//
// This file is part of the Query-based Network Assessment project.
//
// The Query-based Network Assessment is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 of the License.
//
// The Query-based Network Assessment is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along with the
// Query-based Network Assessment; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
//****************************************************************************************//

#include "FileProbe.h"

//****************************************************************************************//
//								FileProbe Class											  //	
//****************************************************************************************//

FileProbe::FileProbe()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	//
	// -----------------------------------------------------------------------

}

FileProbe::~FileProbe()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	//
	// -----------------------------------------------------------------------

}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Public Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
pdVector FileProbe::Run(ProbeData *probeDataIn)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  Determine the type of ProbeData object passed in and call the correct 
	//	Private run method. Only support FilePermissionData and FileAttributeData
	//	types. All others will throw an exception to caller.
	//------------------------------------------------------------------------------------//
    FileAttributeData *fattData = NULL;
    FilePermissionData *fpermisData = NULL;
	string type = typeid(*probeDataIn).name();

	pdVector resultVector;

	if((fattData = dynamic_cast<FileAttributeData*> (probeDataIn)))
	{
	        resultVector = Run(fattData);//(FilePermissionData*)dataIn);

	}else if((fpermisData = dynamic_cast<FilePermissionData*> (probeDataIn)))
	{
		resultVector = Run(fpermisData);

	}else
	{
		throw ProbeException("Error: FileProbe::Run() Invalid ProbeData specified.\n Found type: " + type);
	}

	return resultVector;
}
pdVector FileProbe::Run(FilePermissionData *dataIn)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  Two differnet Run funtions.
	//	Input	: FilePermissionData object.
	//	Output	: pdVector of FilePermissionData objects
	//------------------------------------------------------------------------------------//

	pdVector resultVector;
	FilePermissionData *tmp = NULL;	
	sVector files;

	if(dataIn->path->type == LITTERAL_TYPE)
	{
		//	Create a new data object
		tmp = new FilePermissionData();
		tmp->path->data = dataIn->path->data;
		tmp->SetTestId(dataIn->GetTestId());

		//	Get the file permissions
		GetFilePermissions(tmp);

		//	Add the new data itme to the result vector
		resultVector.push_back(tmp);
		
	}else
	{
		//	Get the matching files
		FindMatchingFiles(dataIn->path->data, &files, "/");
		
		//	Loop through each file contained in the files vector and 
		//	gather file permission information.
		sVector::iterator fIterator;
		for (fIterator=files.begin(); fIterator!=files.end(); fIterator++)
		{
			//	Create a new data object
			tmp = new FilePermissionData();
			tmp->SetTestId(dataIn->GetTestId());
			tmp->path->data = (*fIterator);

			//	Get the file parmissions
			GetFilePermissions(tmp);

			//	Add the new data item to the resultVector
			resultVector.push_back(tmp);
		}

		//	Ensure that at least one object is returned
		if(resultVector.size() == 0)
		{
 			tmp = new FilePermissionData();
			tmp->SetTestId(dataIn->GetTestId());
			tmp->msg = "Unable to find a matching file.";
		}
	}

	return resultVector;
}

pdVector FileProbe::Run(FileAttributeData *dataIn)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  Two differnet Run funtions. 
	//	Input	: FileAttributeData object.
	//	Output	: pdVector of FileAttributeData objects
	//
	//	TO DO:
	//	Test - There are no tests written to test this code...
	//------------------------------------------------------------------------------------//

	pdVector resultVector;
	FileAttributeData *tmp = NULL;
	sVector files;

	if(dataIn->path->type == LITTERAL_TYPE)
	{
		//	Create a new data object
		tmp = new FileAttributeData();
		tmp->path->data = dataIn->path->data;
		tmp->SetTestId(dataIn->GetTestId());
		tmp->nvpPath = dataIn->nvpPath;

		//	Get the file permissions attributes
		GetFileAttributes(tmp);


		//	Add the new data itme to the result vector
		resultVector.push_back(tmp);
	
	}else
	{
		//	Get the matching files
		FindMatchingFiles(dataIn->path->data, &files, "/");
		
		//	Loop through each file contained in the files vector and gather 
		//	file attributes information.
		sVector::iterator fIterator;
		for (fIterator=files.begin(); fIterator!=files.end(); fIterator++)
		{
			//	Create a new data object
			tmp = new FileAttributeData();
			tmp->SetTestId(dataIn->GetTestId());
			tmp->path->data = (*fIterator);
			tmp->nvpPath = dataIn->nvpPath;

			//	Get the file parmissions attributes
			GetFileAttributes(tmp);

			//	Add the new data item to the resultVector
			resultVector.push_back(tmp);
		}
	}

	return resultVector;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  Private Members  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

void FileProbe::GetFileAttributes(FileAttributeData *dataOut)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  Populate the FileAttributeData object
	//
	//------------------------------------------------------------------------------------//

	FILE* fp;
	char buf[1024];

	//////////////////////////////////////////////////////
	/////////////////////  FileType  /////////////////////
	//////////////////////////////////////////////////////

	struct stat sbuf;
	if (stat(dataOut->path->data.c_str(), &sbuf) != 0)
	{
		string errorMessage = "";
		errorMessage.append("(FileProbe) Call to stat failed for file '");
		errorMessage.append(dataOut->path->data);
		errorMessage.append("' - ");
		errorMessage.append(strerror(errno));
		dataOut->msg = errorMessage;
		dataOut->path->data = "";

		return;
	}

	mode_t mode;
	mode = sbuf.st_mode;

	if ((mode & 0xF000) == S_IFIFO) dataOut->type = "fifo";
	else if ((mode & 0xF000) == S_IFCHR) dataOut->type = "character";
	else if ((mode & 0xF000) == S_IFDIR) dataOut->type = "directory";
	else if ((mode & 0xF000) == S_IFBLK) dataOut->type = "block special";
	else if ((mode & 0xF000) == S_IFREG) dataOut->type = "regular";
	else if ((mode & 0xF000) == S_IFLNK) dataOut->type = "symbolic link";
	else if ((mode & 0xF000) == S_IFSOCK) dataOut->type = "socket";

	//////////////////////////////////////////////////////
	/////////////////////  a_time	 /////////////////////
	//////////////////////////////////////////////////////
	dataOut->a_time = Common::ToString((long)sbuf.st_atime); 

	//////////////////////////////////////////////////////
	/////////////////////  c_time	 /////////////////////
	//////////////////////////////////////////////////////
	dataOut->c_time = Common::ToString((long)sbuf.st_atime); 

	//////////////////////////////////////////////////////
	/////////////////////  m_time	 /////////////////////
	//////////////////////////////////////////////////////
	dataOut->m_time = Common::ToString((long)sbuf.st_atime); 

	//////////////////////////////////////////////////////
	////////////////////////  MD5  ///////////////////////
	//////////////////////////////////////////////////////

	fp = fopen(dataOut->path->data.c_str(), "r");
	if (fp == NULL)
	{
		dataOut->md5 = "(error)";
	}
	else
	{
		// Create the md5 hash.  This constructor creates a new md5 object, updates the hash,
		// finalizes the hash, and closes the FILE object.
		
		MD5 context(fp);

		memset(buf, '\0', sizeof(buf));
		snprintf(buf, sizeof(buf)-1, "%s", context.hex_digest());
		buf[sizeof(buf)-1] = '\0';

		dataOut->md5 = buf;
	}

	//////////////////////////////////////////////////////
	//////////////////////  user_id  /////////////////////
	//////////////////////////////////////////////////////

	char uidBuf[16];
        snprintf(uidBuf, sizeof(uidBuf), "%u", sbuf.st_uid);
	dataOut->user_id = uidBuf;

	//////////////////////////////////////////////////////
	/////////////////////// group_id /////////////////////
	//////////////////////////////////////////////////////
	char gidBuf[16];
        snprintf(gidBuf, sizeof(gidBuf), "%u", sbuf.st_gid);
	dataOut->group_id = gidBuf;
}

void FileProbe::GetFilePermissions(FilePermissionData *dataOut)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  opulate the FilePermissionData object
	//
	//------------------------------------------------------------------------------------//
	struct stat sbuf;
	if (stat(dataOut->path->data.c_str(), &sbuf) != 0)
	{
		string errorMessage = "";
		errorMessage.append("(FileProbe) Call to stat failed for file '");
		errorMessage.append(dataOut->path->data);
		errorMessage.append("' - ");
		errorMessage.append(strerror(errno));
		dataOut->msg = errorMessage;
		dataOut->path->data = "";

		return;
	}

	mode_t mode;
	mode = sbuf.st_mode;

	//////////////////////////////////////////////////////
	///////////////////////  SUID  ///////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_ISUID) dataOut->suid = "1";
	else dataOut->suid = "0";

	//////////////////////////////////////////////////////
	///////////////////////  SGID  ///////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_ISGID) dataOut->sgid = "1";
	else dataOut->sgid = "0";

	//////////////////////////////////////////////////////
	//////////////////////  STICKY  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_ISVTX) dataOut->sticky = "1";
 	else dataOut->sticky = "0";

	//////////////////////////////////////////////////////
	///////////////////////  UREAD  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IRUSR) dataOut->uread = "1";
	else dataOut->uread = "0";

	//////////////////////////////////////////////////////
	//////////////////////  UWRITE  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IWUSR) dataOut->uwrite = "1";
	else dataOut->uwrite = "0";

	//////////////////////////////////////////////////////
	///////////////////////  UEXEC  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IXUSR) dataOut->uexec = "1";
	else dataOut->uexec = "0";

	//////////////////////////////////////////////////////
	///////////////////////  GREAD  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IRGRP) dataOut->gread = "1";
	else dataOut->gread = "0";

	//////////////////////////////////////////////////////
	//////////////////////  GWRITE  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IWGRP) dataOut->gwrite = "1";
	else dataOut->gwrite = "0";

	//////////////////////////////////////////////////////
	///////////////////////  GEXEC  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IXGRP) dataOut->gexec = "1";
	else dataOut->gexec = "0";


	//////////////////////////////////////////////////////
	///////////////////////  OREAD  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IROTH) dataOut->oread = "1";
	else dataOut->oread = "0";

	//////////////////////////////////////////////////////
	//////////////////////  OWRITE  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IWOTH) dataOut->owrite = "1";
	else dataOut->owrite = "0";

	//////////////////////////////////////////////////////
	///////////////////////  OEXEC  //////////////////////
	//////////////////////////////////////////////////////

	if (mode & S_IXOTH) dataOut->oexec = "1";
	else dataOut->oexec = "0";

}

void FileProbe::FindMatchingFiles(string pattern, sVector *matches, string fullpath)
{
	//------------------------------------------------------------------------------------//
	//
	//  ABSTRACT
	//
	//  Traverse the file system looking for matching file paths. Each matching path is 
	//	pushed onto the matches vector. 
	//
	//	fullpath may be a file or a directory. 
	//		- if it is a file check if it matches the pattern
	//		- if a directory call self recursively for each name in the directory
	//	
	//------------------------------------------------------------------------------------//


	struct stat statbuf;
	struct dirent *dirp;
	DIR *dp;
	string tmp = "";


	//	Call stat 
	if(lstat(fullpath.c_str(), &statbuf) < 0)
	{
		//	stat error
		//	not sure I want to report this now
		return;
	}

	//	If not a directory check if a match and return
	if(S_ISDIR(statbuf.st_mode) == 0)
	{
		if(myMatcher->IsMatch(pattern.c_str(), fullpath.c_str()))
		  matches->push_back(fullpath);

		return;
	}
	
	///////////////////////////////////////////////////////
	//	It is a directory
	///////////////////////////////////////////////////////

	//	Append a '/'
	if(fullpath.at(fullpath.length()-1) != '/')
	  fullpath.append("/");

	//	Open the directory
	dp = opendir(fullpath.c_str());
	if(dp == NULL)
	{
		//	Error opening directory
		//	not sure this error rmatters
		// cout << "Failed to open the directory" << endl;
		return;
	}

	//	Loop through all names in the directory
	while((dirp = readdir(dp)) != NULL)
	{
		//	Ignore dot and dot-dot
		if(strcmp(dirp->d_name, ".") == 0 || strcmp(dirp->d_name, "..") == 0)
			continue;

		//	append the name after the "/"
		tmp = fullpath;
		tmp.append(dirp->d_name);

		// Nake recursive call
		FindMatchingFiles(pattern, matches, tmp);

	}


	//	Close the directory
	if(closedir(dp) < 0)
	{
		//	Error closing the directory
		//	not sure this error matters
		// cout << "Failed to close the directory" << endl;
		return;
	}
}
