//
// $Id: DOMOvalResultXML.cpp,v 1.3 2004/07/08 15:33:18 bakerj Exp $
//
//************************** Property of the MITRE Corporation ***************************//
//
// Copyright (c) 2003 - The MITRE Corporation
//
// This file is part of the Definition Interpreter project.
//
// The Definition Interpreter is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 of the License.
//
// The Definition Interpreter is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along with the
// Definition Interpreter; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
//****************************************************************************************//


#include "DOMOvalResultXML.h"

//****************************************************************************************//
//								DOMOvalResultXML Class									  //
//****************************************************************************************//

DOMOvalResultXML::DOMOvalResultXML(DOMProcessor *myProcessor, string familyIn)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Create a new DOMOvalResultXML
	//	Initialize all member data. Create the result document..
	// -----------------------------------------------------------------------
	
	//	Set the family
	family = familyIn;

	//	Create the result document
	resultDocument = myProcessor->CreateDOMDocument("oval_results");
	
	//	Get a reference to the oval_result node
	DOMNode * oval_resultNode = DOMCommon::FindNode(resultDocument, "oval_results");
	if(oval_resultNode == NULL)
		throw DOMOvalResultXMLException("Error: Unable to locate the oval_results node in the result document.");

	//	Set namespace info for result document
	DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns", "http://oval.mitre.org/XMLSchema/oval_results");
	DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
	DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns:oval_results", "http://oval.mitre.org/XMLSchema/oval_results");

	if(family.compare("windows") == 0)
	{
		familyXMLNS = "http://oval.mitre.org/XMLSchema/oval_results#windows";
		familyXMLNSQualifier = "windows:";
		xsiSchemaLocation = "http://oval.mitre.org/XMLSchema/oval_results oval-results-schema.xsd http://oval.mitre.org/XMLSchema/oval_results#windows windows-oval-results-schema.xsd";
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns:windows", familyXMLNS.c_str());
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xsi:schemaLocation", xsiSchemaLocation.c_str());

	}else if(family.compare("redhat") == 0)
	{
		familyXMLNS = "http://oval.mitre.org/XMLSchema/oval_results#redhat";
		familyXMLNSQualifier = "redhat:";
		xsiSchemaLocation = "http://oval.mitre.org/XMLSchema/oval_results oval-results-schema.xsd http://oval.mitre.org/XMLSchema/oval_results#redhat redhat-oval-results-schema.xsd";
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns:redhat", familyXMLNS.c_str());
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xsi:schemaLocation", xsiSchemaLocation.c_str());

	}else if(family.compare("solaris") == 0)
	{
		familyXMLNS = "http://oval.mitre.org/XMLSchema/oval_results#solaris";
		familyXMLNSQualifier = "solaris:";
		xsiSchemaLocation = "http://oval.mitre.org/XMLSchema/oval_results oval-results-schema.xsd http://oval.mitre.org/XMLSchema/oval_results#solaris solaris-oval-results-schema.xsd";
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xmlns:redhat", familyXMLNS.c_str());
		DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "xsi:schemaLocation", xsiSchemaLocation.c_str());
	
	}else
	{
		familyXMLNS = "";
		familyXMLNSQualifier = "";
		xsiSchemaLocation = "";
	}

	//	Set the schema version for the data doc
	DOMCommon::AddAttribute((DOMElement*)oval_resultNode, "oval_results:schemaVersion", "1.0");

	//	Add the definitions and tests element to the result document
	DOMElement *definitions = resultDocument->createElement(XMLString::transcode("definitions"));
	DOMElement *tests = resultDocument->createElement(XMLString::transcode("tests"));
	oval_resultNode->appendChild(definitions);
	oval_resultNode->appendChild(tests);
}

DOMOvalResultXML::~DOMOvalResultXML()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	// -----------------------------------------------------------------------
}

// ***************************************************************************************	//
//								Public members												//
// ***************************************************************************************	//
void DOMOvalResultXML::AddCompoundTest(DOMElement *ovalCompoundTest)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Read the oval compound test and create a compound test in the 
	//	result doocument. Copy the id, comment, result, opperator, and 
	//	each subtest. Force compound tests to fall to the bottom of the 
	//	tests element.
	// -----------------------------------------------------------------------
	
	//	Gather the data from the oval test
	string comment = DOMCommon::GetAttributeByName(ovalCompoundTest, "comment");
	string id = DOMCommon::GetAttributeByName(ovalCompoundTest, "id");
	string result = DOMCommon::GetAttributeByName(ovalCompoundTest, "result");
	string operation = DOMCommon::GetDataNodeValue(DOMCommon::FindNode(ovalCompoundTest, "operation"));


	//	Get a reference to the test element in the result document
	DOMNode *tests = DOMCommon::FindNode(resultDocument, "tests");

	//	Create a new compound test element and add it as a child of tests
	DOMElement *test = resultDocument->createElement(XMLString::transcode("compound_test"));
	tests->appendChild(test);

	//	Set the attributes on the on test element
	DOMCommon::AddAttribute(test, "id", id);
	DOMCommon::AddAttribute(test, "comment", comment);
	DOMCommon::AddAttribute(test, "result", result);

	//	Add the operation element to the new compound test
	CreateNodeWithValue(test, "operation", operation);

	//	Copy each subtest from the oval compound test to the new test
	DOMNode *ovalSubtest = NULL;
	for (ovalSubtest = ovalCompoundTest->getFirstChild(); ovalSubtest != NULL; ovalSubtest=ovalSubtest->getNextSibling())
	{
		//	only look at DOMElement nodes
		if(ovalSubtest->getNodeType() == DOMNode::ELEMENT_NODE)
		{
			// only consider subtests
			if(DOMCommon::ToString(ovalSubtest->getNodeName()).compare("subtest")==0)
			{
				AddSubtest(test, (DOMElement*)ovalSubtest);
			}
		}
	}
}

void DOMOvalResultXML::AddDefinition(DOMElement *ovalDefinition)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Read the definition in the oval file and create a new definition
	//	according to the result schema. 
	// -----------------------------------------------------------------------

	//	Get the definition id from the oval definition
	string definitionId = DOMCommon::GetAttributeByName(ovalDefinition, "id");

	//	Get the definition class from the oval definition
	string definitionClass = DOMCommon::GetAttributeByName(ovalDefinition, "class");

	//	Get a reference to the definitions element in the result document
	DOMNode *definitions = DOMCommon::FindNode(resultDocument, "definitions");

	//	Create a new definition element and add it as a child of definitions
	DOMElement *definition = resultDocument->createElement(XMLString::transcode("definition"));
	definitions->appendChild(definition);

	//	Add the id to the definition element
	DOMCommon::AddAttribute(definition, "id", definitionId);

	//	Add the class to the definition element
	DOMCommon::AddAttribute(definition, "class", definitionClass);

	///////////////////////////////////////////////////////
	//	Copy the affected element 
	///////////////////////////////////////////////////////
	//	Only try to copy if the element exists in the oval definition
	DOMNode *ovalAffected = DOMCommon::FindNode(ovalDefinition, "affected");
	
	if (ovalAffected != NULL)
	{
		//	Get the family attribute from the affected element in the oval document
		string ovalAffectedFamily = DOMCommon::GetAttributeByName(ovalAffected, "family");

		//	Create a new affected element and add it as a child of the new definition
		DOMElement *affected = resultDocument->createElement(XMLString::transcode("affected"));
		definition->appendChild(affected);

		//	Add the family attribute to the new affected element
		DOMCommon::AddAttribute(affected, "family", ovalAffectedFamily);

		//	Copy all the platform elements and all the product elements
		DOMNode *ovalProductOrPlatform = NULL;
		for (ovalProductOrPlatform = ovalAffected->getFirstChild(); ovalProductOrPlatform != NULL; ovalProductOrPlatform=ovalProductOrPlatform->getNextSibling())
		{
			//	only look at DOMElement nodes
			if(ovalProductOrPlatform->getNodeType() == DOMNode::ELEMENT_NODE)
			{
				//	Get the element name
				string ovalElementName = DOMCommon::ToString(ovalProductOrPlatform->getNodeName());
				// Is it a platform element
				if(ovalElementName.compare(familyXMLNSQualifier+"platform")==0)
				{
					//	Get the value of the oval platform
					string ovalPlatform = DOMCommon::GetDataNodeValue(ovalProductOrPlatform);

					//	Create a new node and set its value
					CreateNodeWithValue(affected, familyXMLNSQualifier+"platform", ovalPlatform);

				// Is it a product element
				}else if(ovalElementName.compare("product")==0)
				{
					//	Get the value of the oval platform
					string ovalProduct = DOMCommon::GetDataNodeValue(ovalProductOrPlatform);

					//	Create a new node and set its value
					CreateNodeWithValue(affected, "product", ovalProduct);
				}
			}
		}
	}

	///////////////////////////////////////////////////////
	//	Copy the cve element
	///////////////////////////////////////////////////////
	//	Only try to copy if the element exists in the oval definition
	DOMNode *ovalCVE = DOMCommon::FindNode(ovalDefinition, "cveid");
	
	if (ovalCVE != NULL)
	{
		//	Get the status attribute and the value of the cve element
		//	from the oval definition
		string ovalCVEStatus = DOMCommon::GetAttributeByName(ovalCVE, "status");
		string ovalCVEValue = DOMCommon::GetDataNodeValue(ovalCVE);

		//	Create a new cve element and add it as a child of definition and set its value
		DOMElement *cve = CreateNodeWithValue(definition, "cveid", ovalCVEValue);

		//	Add the status attribute to the new cve element
		DOMCommon::AddAttribute(cve, "status", ovalCVEStatus);
	}

	///////////////////////////////////////////////////////
	//	Copy the description element
	///////////////////////////////////////////////////////
	//	Get a reference to the description in the oval definition
	DOMNode *ovalDescription = DOMCommon::FindNode(ovalDefinition, "description");

	//	Get the value of the description from the oval definition
	string ovalDescriptionValue = DOMCommon::GetDataNodeValue(ovalDescription);

	//	Create a new description element and add it as a child of definition and set its value
	CreateNodeWithValue(definition, "description", ovalDescriptionValue);

	///////////////////////////////////////////////////////
	//	Copy the status element
	///////////////////////////////////////////////////////
	//	Get a reference to the status in the oval definition
	DOMNode *ovalStatus = DOMCommon::FindNode(ovalDefinition, "status");

	//	Get the value of the status from the oval definition
	string ovalStatusValue = DOMCommon::GetDataNodeValue(ovalStatus);

	//	Create a new status element and add it as a child of definition and set its value
	CreateNodeWithValue(definition, "status", ovalStatusValue);

	///////////////////////////////////////////////////////
	//	Copy the version element
	///////////////////////////////////////////////////////
	//	Get a reference to the version in the oval definition
	DOMNode *ovalVersion = DOMCommon::FindNode(ovalDefinition, "version");

	//	Get the value of the version from the oval definition
	string ovalVersionValue = DOMCommon::GetDataNodeValue(ovalVersion);

	//	Create a new version element and add it as a child of definition and set its value
	CreateNodeWithValue(definition, "version", ovalVersionValue);

	///////////////////////////////////////////////////////
	//	Copy the criteria element
	///////////////////////////////////////////////////////
	//	Get a reference to the criteria in the oval definition
	DOMNode *ovalCriteria = DOMCommon::FindNode(ovalDefinition, "criteria");

	//	Get the value of the cirteria's result attribute
	string criteriaResult = DOMCommon::GetAttributeByName(ovalCriteria, "result");

	//	Create a new criteria element and add it as a child of definition
	DOMElement *criteria = resultDocument->createElement(XMLString::transcode("criteria"));
	definition->appendChild(criteria);

	//	Set the result attribute on the new criteria element
	DOMCommon::AddAttribute(criteria, "result", criteriaResult);

	//	If the oval definition has a software element copy it to the
	//	result document's criteria
	DOMNode *ovalSoftware = DOMCommon::FindNode(ovalDefinition, "software");

	if(ovalSoftware != NULL)
	{
		//	Get the value of the operation and result attributes 
		//	from the oval software node
		string ovalSoftwareOperation = DOMCommon::GetAttributeByName(ovalSoftware, "operation");
		string ovalSoftwareResult = DOMCommon::GetAttributeByName(ovalSoftware, "result");

		//	Create a new software element and add it as a child of criteria
		DOMElement *software = resultDocument->createElement(XMLString::transcode("software"));
		criteria->appendChild(software);

		//	Add the operation and result attribute to software
		if(ovalSoftwareOperation.compare("") != 0)
			DOMCommon::AddAttribute(software, "operation", ovalSoftwareOperation);
	
		DOMCommon::AddAttribute(software, "result", ovalSoftwareResult);

		//	Add all the subtests to the result document's software element
		AddCriterion(software, (DOMElement*)ovalSoftware);
	}

	//	If the oval definition has a configuration element copy it to the
	//	result document's criteria
	DOMNode *ovalCofiguration = DOMCommon::FindNode(ovalDefinition, "configuration");

	if(ovalCofiguration != NULL)
	{
		//	Get the value of the operation and result attributes 
		//	from the oval configuration node
		string ovalConfigurationOperation = DOMCommon::GetAttributeByName(ovalCofiguration, "operation");
		string ovalConfigurationResult = DOMCommon::GetAttributeByName(ovalCofiguration, "result");

		//	Create a new configuration element and add it as a child of criteria
		DOMElement *configuration = resultDocument->createElement(XMLString::transcode("configuration"));
		criteria->appendChild(configuration);

		//	Add the operation and result attribute to configuration
		if(ovalConfigurationOperation.compare("") != 0)
			DOMCommon::AddAttribute(configuration, "operation", ovalConfigurationOperation);
	
		DOMCommon::AddAttribute(configuration, "result", ovalConfigurationResult);

		//	Add all the subtests to the result document's configuration element
		AddCriterion(configuration, (DOMElement*)ovalCofiguration);
	}
}

void DOMOvalResultXML::AddHostInfo(string hostName, string ip)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Add the host_name and ip_address elements to the result file
	// -----------------------------------------------------------------------

	//	Get the root of the result file
	DOMNode *oval_resultNode = DOMCommon::FindNode(resultDocument, "oval_results");
	if(oval_resultNode == NULL)
		throw DOMOvalResultXMLException("Error: Unable to locate the oval_results node in the result document.");


	//	Create the host_name element and set its value
	DOMElement *hnElem = resultDocument->createElement(XMLString::transcode("host_name"));
	DOMText *hnText = resultDocument->createTextNode(XMLString::transcode(hostName.c_str()));
	hnElem->appendChild(hnText);

	//	Create the ip_address element and set its value
	DOMElement *ipElem = resultDocument->createElement(XMLString::transcode("ip_address"));
	DOMText *ipText = resultDocument->createTextNode(XMLString::transcode(ip.c_str()));
	ipElem->appendChild(ipText);


	//	Add the elements to the the result document - before the definitions element
	oval_resultNode->insertBefore(ipElem, (DOMElement*)DOMCommon::FindNode(resultDocument,"definitions"));
	oval_resultNode->insertBefore(hnElem, ipElem);
}

void DOMOvalResultXML::AddTest(DOMElement *ovalTest)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Read the oval test and add a test to the result document.
	//	Copy the id, comment, and result. Add the correct namespace attribute 
	//	Force tests to be earlier siblings to compound tests.
	// -----------------------------------------------------------------------

	//	Gather the data from the oval test
	string testName = DOMCommon::ToString(ovalTest->getNodeName());
	string comment = DOMCommon::GetAttributeByName(ovalTest, "comment");
	string id = DOMCommon::GetAttributeByName(ovalTest, "id");
	string result = DOMCommon::GetAttributeByName(ovalTest, "result");

	//	Get a reference to the test element in the result document
	DOMNode *tests = DOMCommon::FindNode(resultDocument, "tests");

	//	Create a new test element and add it as the first child of tests
	DOMElement *test = resultDocument->createElement(XMLString::transcode(testName.c_str()));
	tests->insertBefore(test, tests->getFirstChild());

	//	Set the attributes on the on test element
	DOMCommon::AddAttribute(test, "id", id);
	DOMCommon::AddAttribute(test, "comment", comment);
	DOMCommon::AddAttribute(test, "result", result);
	//	unknown_tests are the only test currently defined in the oval_results schema
	//	so no need to add a xmlns attribute.
	if(testName.compare("unknown_test") != 0)
		DOMCommon::AddAttribute(test, "xmlns", familyXMLNS);

}

void DOMOvalResultXML::AddTimeStamps(DOMElement *dataTimeStamp, string ovalTimeStamp)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Create a time stamp for the result file, copy the time stamp
	//	from the data file, and use the input oval timestamp.
	//	Parse the dataTimeStamp element to get its value
	//	for use as the data time stamp.
	// -----------------------------------------------------------------------

	//	Get the current time
	string time = Common::GetTimeStamp();

	//	Get the root of the result file
	DOMNode *oval_resultNode = DOMCommon::FindNode(resultDocument, "oval_results");
	if(oval_resultNode == NULL)
		throw DOMOvalResultXMLException("Error: Unable to locate the oval_results node in the result document.");

	//	Get the data document's time stamp value
	string strDataTimeStamp = DOMCommon::GetDataNodeValue(dataTimeStamp);

	//	Create the time_stamps element and add it to the results document
	DOMElement *timesElem = resultDocument->createElement(XMLString::transcode("time_stamps"));
	oval_resultNode->insertBefore(timesElem, oval_resultNode->getFirstChild());

	//	Create the oval_definitions element and set its value
	DOMElement *ovalTimeElem = resultDocument->createElement(XMLString::transcode("oval_definitions"));
	if(ovalTimeStamp.compare("") != 0)
	{
		DOMText *ovalTimeText = resultDocument->createTextNode(XMLString::transcode(strDataTimeStamp.c_str()));
		ovalTimeElem->appendChild(ovalTimeText);
	}

	//	Create the system_characteristics element and set its value
	DOMElement *dataTimeElem = resultDocument->createElement(XMLString::transcode("system_characteristics"));
	DOMText *dataTimeText = resultDocument->createTextNode(XMLString::transcode(strDataTimeStamp.c_str()));
	dataTimeElem->appendChild(dataTimeText);

	//	Create the results element and set its value
	DOMElement *resultsElem = resultDocument->createElement(XMLString::transcode("results"));
	DOMText *resultsText = resultDocument->createTextNode(XMLString::transcode(time.c_str()));
	resultsElem->appendChild(resultsText);

	//	Add the elements to the the result document
	timesElem->appendChild(ovalTimeElem);
	timesElem->appendChild(dataTimeElem);
	timesElem->appendChild(resultsElem);
}


XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* DOMOvalResultXML::GetResultFile()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Return the resultDocument
	// -----------------------------------------------------------------------

	return resultDocument;
}

// ***************************************************************************************	//
//								Private members												//
// ***************************************************************************************	//
void DOMOvalResultXML::AddCriterion(DOMElement *newCriterionParent, DOMElement *ovalCriterionParent)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Copy each criterion in the ovalCriterion parent to the 
	//	newCriterionParent. 
	// -----------------------------------------------------------------------
	DOMNode *ovalCriterion	= NULL;
	DOMElement *newCriterion	= NULL;
	


	for (ovalCriterion = ovalCriterionParent->getFirstChild(); ovalCriterion != NULL; ovalCriterion=ovalCriterion->getNextSibling())
	{
		//	only look at DOMElement nodes
		if(ovalCriterion->getNodeType() == DOMNode::ELEMENT_NODE)
		{
			//	Get the attributes from the oval criterion
			string ovalComment = DOMCommon::GetAttributeByName(ovalCriterion, "comment");
			string ovalNegate = DOMCommon::GetAttributeByName(ovalCriterion, "negate");
			string ovalResult = DOMCommon::GetAttributeByName(ovalCriterion, "result");
			string ovalTest_Ref = DOMCommon::GetAttributeByName(ovalCriterion, "test_ref");
			
			//	Create a new criterion element and add it to newCriterionParent
			newCriterion = resultDocument->createElement(XMLString::transcode("criterion"));
			newCriterionParent->appendChild(newCriterion);

			//	Add the attributes to the new criterion
			DOMCommon::AddAttribute(newCriterion, "comment", ovalComment);
			if(ovalNegate.compare("true") == 0)
				DOMCommon::AddAttribute(newCriterion, "negate", ovalNegate);

			
			DOMCommon::AddAttribute(newCriterion, "result", ovalResult);			
			DOMCommon::AddAttribute(newCriterion, "test_ref", ovalTest_Ref);
		}
	}
}

void DOMOvalResultXML::AddSubtest(DOMElement *newCompoundTest, DOMElement *ovalSubtest)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Create a new subtest element and add it as a child of newCompoundTest.
	//	Copy id, comment, negate, and result from ovalSubtest
	// -----------------------------------------------------------------------

	//	Gather data from ovalSubtest
	string ovalComment	= DOMCommon::GetAttributeByName(ovalSubtest, "comment");
	string ovalTest_Ref	= DOMCommon::GetAttributeByName(ovalSubtest, "test_ref");
	string ovalResult	= DOMCommon::GetAttributeByName(ovalSubtest, "result");
	string ovalNegate	= DOMCommon::GetAttributeByName(ovalSubtest, "negate");

	//	Create a new subtest element and add it as a child of newCompoundTest
	DOMElement *newSubtest = resultDocument->createElement(XMLString::transcode("subtest"));
	newCompoundTest->appendChild(newSubtest);

	//	Set attributes on the new subtest
	DOMCommon::AddAttribute(newSubtest, "test_ref", ovalTest_Ref);

	if(ovalNegate.compare("true") == 0)
		DOMCommon::AddAttribute(newSubtest, "negate", ovalNegate);

	
	DOMCommon::AddAttribute(newSubtest, "result", ovalResult);
	
}

DOMElement* DOMOvalResultXML::CreateNodeWithValue(DOMElement *parent, string name, string value)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Create a new DOMElement named "name" add to it a new DOMText node 
	//	with value "value' and add the new DOMElement as a child of parent
	// -----------------------------------------------------------------------

	//	Create the new node
	DOMElement *newNode = resultDocument->createElement(XMLString::transcode(name.c_str()));

	//	Create a new test node and append it to newNode
	DOMText *newText = resultDocument->createTextNode(XMLString::transcode(value.c_str()));
	newNode->appendChild(newText);

	//	Append it is a child of parent
	parent->appendChild(newNode);

	return newNode;

}

//****************************************************************************************//
//							DOMOvalResultXMLException Class								  //	
//****************************************************************************************//
DOMOvalResultXMLException::DOMOvalResultXMLException() : Exception()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Default constructor simply set the severity to ERROR_FATAL. This is 
	//	done with the explicit call to the Exception class default constructor.
	//
	// -----------------------------------------------------------------------

}

DOMOvalResultXMLException::DOMOvalResultXMLException(string errMsgIn) : Exception(errMsgIn)
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Set the error message and then set the severity to ERROR_FATAL. This is 
	//	done with the explicit call to the Exception class constructor that 
	//	takes a single string param.
	//
	// -----------------------------------------------------------------------

}

DOMOvalResultXMLException::~DOMOvalResultXMLException()
{
	// -----------------------------------------------------------------------
	//	Abstract
	//
	//	Do nothing for now
	//
	// -----------------------------------------------------------------------

}

